const PROVIDER_CODES = {
    "PG": "pg",
    "PGSOFT": "pg",
    "PRAGMATIC": "pp",
    "HABANERO": "hb",
    "JOKER": "jk",
    "MICROGAMING": "mg",
    "SPADEGAMING": "sg",
    "PLAYTECH": "pt",
    "CQ9": "cq9",
    "NETENT": "nt",
    "NOLIMIT": "nl",
    "JILI": "jl",
    "PLAYSTAR": "ps",
    "IDNSLOT": "idn",
    "BIGTIMEGAMING": "btg"
};

function detectProvider() {
    const title = document.title.toUpperCase();
    for (const provider in PROVIDER_CODES) {
        if (title.includes(provider)) {
            return provider;
        }
    }
    
    const url = window.location.pathname.toLowerCase();
    for (const provider in PROVIDER_CODES) {
        const code = PROVIDER_CODES[provider].toLowerCase();
        if (url.includes(code) || url.includes(provider.toLowerCase())) {
            return provider;
        }
    }
    
    const firstCard = document.querySelector('.rtp-card');
    if (firstCard) {
        const provCode = firstCard.getAttribute('data-prov');
        if (provCode) {
            for (const provider in PROVIDER_CODES) {
                if (PROVIDER_CODES[provider].toLowerCase() === provCode.toLowerCase()) {
                    return provider;
                }
            }
        }
    }
    
    return "PG";
}

const CURRENT_PROVIDER = typeof SERVER_PROVIDER !== 'undefined' ? SERVER_PROVIDER : detectProvider();
const providerCode = PROVIDER_CODES[CURRENT_PROVIDER] || CURRENT_PROVIDER.toLowerCase();

function hashCode(str) {
    let hash = 0;
    for (let i = 0; i < str.length; i++) {
        const char = str.charCodeAt(i);
        hash = ((hash << 5) - hash) + char;
        hash = hash & hash;
    }
    return Math.abs(hash);
}

function generateRTP(gameIndex) {
    const time = Math.floor(Date.now() / (30 * 60 * 1000));
    const seed = hashCode(`${CURRENT_PROVIDER}-${gameIndex}-${time}`);
    const rng = seed % 100;

    if (rng < 5) {

        return 30 + (seed % 10);
    } else if (rng < 15) {

        return 40 + (seed % 20);
    } else if (rng < 50) {

        return 60 + (seed % 21); 
    } else {
        return 81 + (seed % 18); 
    }
}



function getRTPColor(rtpValue) {
    if (rtpValue >= 75) {
        return 'great';
    } else if (rtpValue >= 40) {
        return 'good';
    } else {
        return 'bad';
    }
}

function generateJamGacor(gameIndex, rtpValue) {
    const now = new Date();
    const hour = now.getHours();
    const date = now.getDate();
    
    const target = rtpValue % 6;
    
    const hour24 = (hour + 7) % 24;
    const randomTarget = String((hour24 + target) % 24).padStart(2, "0");
    
    const min1 = String(rtpValue % 60).padStart(2, "0");
    const min2 = String((rtpValue * date) % 60).padStart(2, "0");
    const hour2Display = String(hour24).padStart(2, "0");
    
    return `<i class="lni lni-alarm-clock"></i> Jam: ${hour2Display}:${min1} - ${randomTarget}:${min2}`;
}

function generatePolaPatterns(gameIndex, rtpValue) {
    const patterns = [];
    
    if (rtpValue >= 30) {
        const target = rtpValue > 70 ? (rtpValue % 6) + (rtpValue % 6) : (rtpValue % 6) + 8;
        const n1 = gameIndex % 2;
        const n2 = target % 6;
        const n3 = gameIndex % 5;
        
        const turbo = [
            "<span class='text-success fw-bold'>Turbo On</span> <i class='lni lni-checkmark-circle fw-bold text-success'></i>",
            "<span class='text-danger fw-bold'>Turbo Off</span> <i class='lni lni-cross-circle fw-bold text-danger'></i>"
        ];
        
        const patternTypes = ["Auto", "Manual 3", "Manual 5", "Manual 7", "Manual 9"];
        const spinValues = [10, 20, 30, 50, 70, 100];
        
        for (let i = 0; i < 3; i++) {
            const p1 = (Math.floor(n2 + (gameIndex * n3 + rtpValue * i)) + n3) % 2;
            const p3 = Math.floor(i + n3 * i) * rtpValue % 5;
            const p2 = (Math.floor(i + (i * n2 + rtpValue)) + n1) % 6;
            
            let patternRow = "";
            if (patternTypes[p3] === "Auto") {
                patternRow = `<td>${patternTypes[p3]} ${spinValues[p2]}</td><td>${turbo[p1]}</td>`;
            } else {
                patternRow = `<td>${patternTypes[p3]}</td><td>${turbo[p1]}</td>`;
            }
            
            patterns.push(patternRow);
        }
    } else {
        patterns.push("<td class='text-danger fw-bold'>Pola Tidak Tersedia!!</td>");
        patterns.push("<td>Tidak Disarankan Bermain</td>");
        patterns.push("<td>Game ini</td>");
    }
    
    return patterns;
}

function checkHotGame(gameIndex, rtpValue) {
    if (gameIndex <= 8) {
        return;
    }
    
    if (rtpValue >= 80) {
        const card = document.querySelector(`[id^="percent-txt-${gameIndex}"]`)?.closest('.rtp-card');
        if (card) {
            const imgContainer = card.querySelector('.place-img-rtp');
            if (imgContainer && !imgContainer.querySelector('.top-game') && !imgContainer.querySelector('.hot-game')) {
                const hotTag = document.createElement('div');
                hotTag.className = 'hot-game';
                imgContainer.appendChild(hotTag);
            }
        }
    }
}

function initializeGames() {
    const rtpCards = document.querySelectorAll('.rtp-card');
    
    rtpCards.forEach(card => {
        const elements = card.querySelectorAll('[id*="-"]');
        let gameIndex = null;
        
        for (const element of elements) {
            const idParts = element.id.split('-');
            const lastPart = idParts[idParts.length - 1];
            if (!isNaN(parseInt(lastPart))) {
                gameIndex = parseInt(lastPart);
                break;
            }
        }
        
        if (!gameIndex) return;
        
        const percentTxt = card.querySelector(`[id^="percent-txt-"]`);
        const percentBar = card.querySelector(`[id^="percent-bar-"]`);
        const polaSlot1 = card.querySelector(`[id^="pola-slot-1-"]`);
        const polaSlot2 = card.querySelector(`[id^="pola-slot-2-"]`);
        const polaSlot3 = card.querySelector(`[id^="pola-slot-3-"]`);
        const jamGacor = card.querySelector(`[id^="jam-gacor-"]`);
        
        const rtpValue = generateRTP(gameIndex);
        const rtpColorClass = getRTPColor(rtpValue);
        
        if (percentTxt) {
            percentTxt.innerText = rtpValue + "%";
        }
        
        if (percentBar) {
            percentBar.setAttribute("aria-valuenow", rtpValue);
            percentBar.style.width = rtpValue + "%";
            percentBar.classList.remove('great', 'good', 'bad');
            percentBar.classList.add(rtpColorClass);
        }
        
        const patterns = generatePolaPatterns(gameIndex, rtpValue);
        if (polaSlot1 && patterns[0]) polaSlot1.innerHTML = patterns[0];
        if (polaSlot2 && patterns[1]) polaSlot2.innerHTML = patterns[1];
        if (polaSlot3 && patterns[2]) polaSlot3.innerHTML = patterns[2];
        
        if (jamGacor) {
            jamGacor.innerHTML = generateJamGacor(gameIndex, rtpValue);
        }

        checkHotGame(gameIndex, rtpValue);
    });
}

function updateDate() {
    const rtpDateElement = document.getElementById("rtp-date");
    if (rtpDateElement) {
        const options = { weekday: 'long', year: 'numeric', month: 'long', day: 'numeric' };
        const today = new Date();
        const formattedDate = today.toLocaleDateString("en-US", options);
        rtpDateElement.textContent = formattedDate;
    }
}

document.addEventListener('DOMContentLoaded', function() {
    updateDate();
    initializeGames();
    
    setInterval(function() {
        initializeGames();
    }, 30 * 60 * 1000);
});

function goUp() {
    document.body.scrollTop = 0;
    document.documentElement.scrollTop = 0;
}
const swiper = new Swiper(".slider", {
    direction: "horizontal",
    centeredSlides: true,
    loop: true,
    autoplay: {
      delay: 2800
    },
    effect: "flip",
    flipEffect: {
      slideShadows: false
    },
    pagination: {
      el: ".swiper-pagination"
    },
    breakpoints: {
      320: {
        spaceBetween: 1
      },
      480: {
        spaceBetween: 2
      },
      640: {
        spaceBetween: 3
      }
    },
    navigation: {
      nextEl: ".swiper-button-next",
      prevEl: ".swiper-button-prev"
    }
  });
  placeImg = $(".place-img-rtp").hover(function () {
    $(this).find(".btn-play").fadeIn(100);
    $(this).find(".rtp-card-img").css({
      filter: "grayscale(100%)",
      opacity: "0.5",
      transform: "scale(1.1)",
      transition: "all .3s ease",
      "vertical-align": "middle"
    });
  }, function () {
    $(this).find(".btn-play").fadeOut(100);
    $(this).find(".rtp-card-img").css({
      filter: "grayscale(0%)",
      opacity: "1",
      transform: "scale(1)",
      transition: "all .2s ease",
      "vertical-align": "middle"
    });
  });

  $(function () {
    $(".lazy").Lazy({
      effect: "fadeIn"
    });

  });