<?php
require_once 'config-db.php';

function generateGameCard($provider, $gameNumber, $daftarLink = null) {
    if ($daftarLink === null) {
        $daftarLink = getSetting('daftar_link', 'https://qoolink.co/bangunwin');
    }

    $providerCode = getProviderCode($provider);

    // Get game from database
    $pdo = getDB();
    $stmt = $pdo->prepare("SELECT game_name, game_image_path FROM games WHERE provider_code = ? AND is_active = 1 ORDER BY display_order ASC, id DESC LIMIT 1 OFFSET ?");
    $stmt->execute([strtoupper($provider), $gameNumber - 1]);
    $game = $stmt->fetch();

    // Fallback to old system if no game in DB
    if (!$game) {
        $imagePath = 'img/games-' . strtolower($providerCode) . '/' . strtolower($providerCode) . '-' . $gameNumber . '.png';
        $gameName = 'Game ' . $gameNumber;
    } else {
        $imagePath = $game['game_image_path'];
        $gameName = $game['game_name'];
    }

    $isTop = ($gameNumber <= 8);

    $seed = crc32($provider . $gameNumber . date('Ymd'));
    mt_srand($seed);
    $rtpValue = mt_rand(10, 98);

    $gameTag = '';
    if ($isTop) {
        $gameTag = '<div class="top-game"></div>';
    } elseif ($rtpValue >= 80) {
        $gameTag = '<div class="hot-game"></div>';
    }

    $html = '
    <div class="col-lg-4 col-6">
        <div data-prov="' . strtolower($providerCode) . '" class="rtp-card animate__animated animate__bounceIn">
            <div class="row g-1">
                <div class="col-lg-6 align-self-center">
                    <div class="place-img-rtp">
                        ' . $gameTag . '
                        <button onclick="location.href=\'' . $daftarLink . '\';" class="btn-play shadow"><i class="lni lni-heart-fill"></i> Mari Bermain</button>
                        <img class="lazy shadow rtp-card-img" data-src="' . $imagePath . '" src="data:image/webp;base64,UklGRiIAAABXRUJQVlA4TAAAAAAvQWxvAGs=" alt="' . htmlspecialchars($gameName) . '" onerror="this.style.display=\'none\'; this.nextElementSibling.style.display=\'flex\';">
                        <div class="game-img-placeholder" style="display: none;">
                        </div>
                    </div>
                </div>
                <div class="col-lg-6">
                    <div class="pola-wrapper shadow text-center">
                        <div class="pola-title-wrapper">
                        <div class="pola-title-row">
                            <h4><i class="lni lni-bar-chart text-white"></i> Pola Slot:</h4>
                            <div class="icon-providers">
                                <img src="img/icons/' . strtolower($providerCode) . '.png" alt="icon provider" onerror="this.style.display=\'none\'">
                            </div>
                            </div>
                        </div>
                        <hr>
                        <table class="table-pola text-center">
                            <tbody>
                                <tr id="pola-slot-1-' . $gameNumber . '"></tr>
                                <tr id="pola-slot-2-' . $gameNumber . '"></tr>
                                <tr id="pola-slot-3-' . $gameNumber . '"></tr>
                            </tbody>
                        </table>
                        <h5 class="mt-3 fw-bold" id="jam-gacor-' . $gameNumber . '"></h5>
                        <div class="percent">
                            <p id="percent-txt-' . $gameNumber . '" style="z-index: 15">00%</p>
                            <div id="percent-bar-' . $gameNumber . '" class="percent-bar" role="progressbar" aria-valuemin="0" aria-valuemax="100" aria-valuenow="" style="width: 0"></div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>';

    return $html;
}

function generateGameGrid($provider, $limit = 20, $offset = 0, $daftarLink = null) {
    if ($daftarLink === null) {
        $daftarLink = getSetting('daftar_link', 'https://qoolink.co/bangunwin');
    }

    $html = '';
    $total = getTotalGameCount($provider);
    $start = $offset + 1;
    $end = min($offset + $limit, $total);

    for ($i = $start; $i <= $end; $i++) {
        $html .= generateGameCard($provider, $i, $daftarLink);
    }

    return $html;
}

function getTotalGameCount($provider) {
    $pdo = getDB();
    $providerCode = strtoupper($provider);

    $stmt = $pdo->prepare("SELECT total_games FROM providers WHERE provider_code = ? AND is_active = 1 LIMIT 1");
    $stmt->execute([$providerCode]);
    $result = $stmt->fetch();

    if ($result && $result['total_games'] > 0) {
        return (int)$result['total_games'];
    }

    $gameCounts = [
        'PP' => 555,
        'PG' => 99,
        'CQ9' => 236,
        'JL' => 199,
        'JK' => 218,
        'MG' => 353,
        'SG' => 115,
        '5G' => 32,
        'LIVE22' => 45,
        'PS' => 77,
        'SBO' => 96,
        'FS' => 57,
        'AP' => 42,
        'HB' => 183,
        'HS' => 109,
        'NLC' => 107,
    ];

    $key = strtoupper($provider);
    return $gameCounts[$key] ?? 100;
}

function getProviderCode($provider) {
    $providerCodes = [
        'PG_SLOTS' => 'pg',
        'PRAGMATIC_PLAY' => 'pp',
        'HABANERO' => 'hb',
        'JOKER' => 'jk',
        'MICROGAMING' => 'mg',
        'SPADE_GAMING' => 'sg',
        'CQ9' => 'cq9',
        'JILI' => 'jl',
        'PLAYSTAR' => 'ps',
        'LIVE_22' => 'live22',
        'NO_LIMIT_CITY' => 'nlc',
        'FAST_SPIN' => 'fs',
        'ADVANTPLAY' => 'ap',
        'HACKSAW' => 'hs',
        'SBO_SLOT' => 'sbo',
        '5G_GAMES' => '5g',
    ];

    $providerUpper = strtoupper(str_replace('-', '_', $provider));
    return isset($providerCodes[$providerUpper]) ? $providerCodes[$providerUpper] : strtolower($provider);
}

function getCurrentProvider() {
    $filename = basename($_SERVER['PHP_SELF'], '.php');

    $providerMap = [
        'pgsoft' => 'PG_SLOTS',
        'pragmatic' => 'PRAGMATIC_PLAY',
        'habanero' => 'HABANERO',
        'joker' => 'JOKER',
        'microgaming' => 'MICROGAMING',
        'spadegaming' => 'SPADE_GAMING',
        'cq9' => 'CQ9',
        'jili' => 'JILI',
        'playstar' => 'PLAYSTAR',
        'live_22' => 'LIVE_22',
        'nolimit' => 'NO_LIMIT_CITY',
        'fastspin' => 'FAST_SPIN',
        'advantplay' => 'ADVANTPLAY',
        'hacksaw' => 'HACKSAW',
        'sbo' => 'SBO_SLOT',
        '5G_GAMES' => '5G_GAMES',
    ];

    return isset($providerMap[$filename]) ? $providerMap[$filename] : 'PG_SLOTS';
}

function getAllProviders() {
    $pdo = getDB();
    $stmt = $pdo->query("SELECT provider_code, provider_name, provider_image FROM providers WHERE is_active = 1 ORDER BY display_order ASC, provider_name ASC");
    $providers = $stmt->fetchAll();

    // Convert to format yang dipakai di index.php
    $result = [];

    foreach ($providers as $prov) {
        $code = $prov['provider_code'];
        $result[] = [
            'name' => $prov['provider_name'],
            'file' => $prov['provider_image'],
            'link' => '/' . strtolower($code)
        ];
    }

    return $result;
}
