<?php
/**
 * USER MANAGEMENT PAGE
 * CRUD admin users
 */

define('ADMIN_ACCESS', true);
require_once 'config/database.php';
require_once 'config/functions.php';

startSecureSession();
requireLogin();

$pdo = getDBConnection();
$error = '';
$success = '';

// Handle Add User
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['add_user'])) {
    $username = sanitize($_POST['username'] ?? '');
    $email = sanitize($_POST['email'] ?? '');
    $fullName = sanitize($_POST['full_name'] ?? '');
    $password = $_POST['password'] ?? '';
    $confirmPassword = $_POST['confirm_password'] ?? '';

    if (empty($username) || empty($email) || empty($password)) {
        $error = 'Username, email, dan password harus diisi!';
    } elseif (strlen($username) < 3) {
        $error = 'Username minimal 3 karakter!';
    } elseif (!isValidEmail($email)) {
        $error = 'Format email tidak valid!';
    } elseif (strlen($password) < 6) {
        $error = 'Password minimal 6 karakter!';
    } elseif ($password !== $confirmPassword) {
        $error = 'Password dan konfirmasi tidak cocok!';
    } else {
        try {
            $hashedPassword = hashPassword($password);
            $stmt = $pdo->prepare("INSERT INTO admin_users (username, email, full_name, password, is_active, created_at)
                                   VALUES (?, ?, ?, ?, 1, NOW())");
            $stmt->execute([$username, $email, $fullName, $hashedPassword]);

            setFlashMessage('success', 'User berhasil ditambahkan!');
            header('Location: users.php');
            exit;
        } catch (PDOException $e) {
            if ($e->getCode() == 23000) {
                $error = 'Username atau email sudah digunakan!';
            } else {
                $error = 'Terjadi kesalahan: ' . $e->getMessage();
            }
        }
    }
}

// Handle Edit User
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['edit_user'])) {
    $userId = intval($_POST['user_id']);
    $fullName = sanitize($_POST['full_name'] ?? '');
    $email = sanitize($_POST['email'] ?? '');
    $isActive = isset($_POST['is_active']) ? 1 : 0;

    // Prevent deactivating yourself
    if ($userId == $_SESSION['admin_user_id'] && $isActive == 0) {
        $error = 'Tidak bisa menonaktifkan akun sendiri!';
    } elseif (empty($email)) {
        $error = 'Email harus diisi!';
    } elseif (!isValidEmail($email)) {
        $error = 'Format email tidak valid!';
    } else {
        try {
            $stmt = $pdo->prepare("UPDATE admin_users SET full_name = ?, email = ?, is_active = ? WHERE id = ?");
            $stmt->execute([$fullName, $email, $isActive, $userId]);

            setFlashMessage('success', 'User berhasil diupdate!');
            header('Location: users.php');
            exit;
        } catch (PDOException $e) {
            if ($e->getCode() == 23000) {
                $error = 'Email sudah digunakan oleh user lain!';
            } else {
                $error = 'Terjadi kesalahan: ' . $e->getMessage();
            }
        }
    }
}

// Handle Reset Password
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['reset_password'])) {
    $userId = intval($_POST['user_id']);
    $newPassword = $_POST['new_password'] ?? '';
    $confirmPassword = $_POST['confirm_password'] ?? '';

    if (empty($newPassword) || empty($confirmPassword)) {
        $error = 'Password harus diisi!';
    } elseif (strlen($newPassword) < 6) {
        $error = 'Password minimal 6 karakter!';
    } elseif ($newPassword !== $confirmPassword) {
        $error = 'Password dan konfirmasi tidak cocok!';
    } else {
        $hashedPassword = hashPassword($newPassword);
        $stmt = $pdo->prepare("UPDATE admin_users SET password = ? WHERE id = ?");
        $stmt->execute([$hashedPassword, $userId]);

        setFlashMessage('success', 'Password berhasil direset!');
        header('Location: users.php');
        exit;
    }
}

// Handle Delete User
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['delete_user'])) {
    $userId = intval($_POST['user_id']);

    // Prevent deleting yourself
    if ($userId == $_SESSION['admin_user_id']) {
        setFlashMessage('error', 'Tidak bisa menghapus akun sendiri!');
    } else {
        $stmt = $pdo->prepare("DELETE FROM admin_users WHERE id = ?");
        $stmt->execute([$userId]);

        setFlashMessage('success', 'User berhasil dihapus!');
    }
    header('Location: users.php');
    exit;
}

// Get action and ID
$action = $_GET['action'] ?? 'list';
$userId = isset($_GET['id']) ? intval($_GET['id']) : 0;

// Get user for edit
$userEdit = null;
if ($action === 'edit' && $userId) {
    $stmt = $pdo->prepare("SELECT * FROM admin_users WHERE id = ?");
    $stmt->execute([$userId]);
    $userEdit = $stmt->fetch();

    if (!$userEdit) {
        header('Location: users.php');
        exit;
    }
}

// Get all users
$users = $pdo->query("SELECT * FROM admin_users ORDER BY created_at DESC")->fetchAll();

$pageTitle = 'User Management';
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= $pageTitle ?> - Admin Panel</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://cdn.jsdelivr.net/npm/remixicon@3.5.0/fonts/remixicon.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Saira:wght@400;500;600;700&family=Poppins:wght@300;400;500;600&display=swap" rel="stylesheet">
    <style>
        * { font-family: 'Poppins', sans-serif; }
        h1, h2, h3 { font-family: 'Saira', sans-serif; }
    </style>
</head>
<body class="bg-gray-50">

    <?php require_once 'includes/navbar.php'; ?>

    <main class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">

        <!-- Page Header -->
        <div class="mb-8 flex justify-between items-center">
            <div>
                <h1 class="text-3xl font-bold text-gray-900 flex items-center">
                    <i class="ri-team-line text-blue-600 mr-3"></i>
                    User Management
                </h1>
                <p class="text-gray-600 mt-2">Manage admin users and permissions</p>
            </div>
            <?php if ($action === 'list'): ?>
            <a href="?action=add" class="px-6 py-3 bg-blue-600 text-white rounded-lg font-semibold hover:bg-blue-700 transition-all shadow-lg">
                <i class="ri-user-add-line mr-2"></i>Add New User
            </a>
            <?php endif; ?>
        </div>

        <!-- Flash Messages -->
        <?php if ($message = getFlashMessage()): ?>
            <div class="mb-6 p-4 rounded-lg border-l-4 <?= $message['type'] === 'success' ? 'bg-green-50 border-green-500 text-green-800' : 'bg-red-50 border-red-500 text-red-800' ?>">
                <div class="flex items-center">
                    <i class="<?= $message['type'] === 'success' ? 'ri-checkbox-circle-line' : 'ri-error-warning-line' ?> text-2xl mr-3"></i>
                    <p class="font-medium"><?= htmlspecialchars($message['message']) ?></p>
                </div>
            </div>
        <?php endif; ?>

        <?php if ($error): ?>
            <div class="mb-6 p-4 rounded-lg border-l-4 bg-red-50 border-red-500 text-red-800">
                <div class="flex items-center">
                    <i class="ri-error-warning-line text-2xl mr-3"></i>
                    <p class="font-medium"><?= htmlspecialchars($error) ?></p>
                </div>
            </div>
        <?php endif; ?>

        <?php if ($action === 'list'): ?>

            <!-- User List -->
            <div class="bg-white rounded-xl shadow-sm border border-gray-200 overflow-hidden">
                <div class="overflow-x-auto">
                    <table class="min-w-full divide-y divide-gray-200">
                        <thead class="bg-gray-50">
                            <tr>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">ID</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">User</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Email</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Last Login</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Actions</th>
                            </tr>
                        </thead>
                        <tbody class="bg-white divide-y divide-gray-200">
                            <?php foreach ($users as $user): ?>
                            <tr class="hover:bg-gray-50 transition">
                                <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                                    #<?= $user['id'] ?>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <div class="flex items-center">
                                        <div class="w-10 h-10 rounded-full bg-blue-600 flex items-center justify-center text-white font-bold mr-3">
                                            <?= strtoupper(substr($user['username'], 0, 1)) ?>
                                        </div>
                                        <div>
                                            <div class="text-sm font-medium text-gray-900">
                                                <?= htmlspecialchars($user['full_name'] ?: $user['username']) ?>
                                                <?php if ($user['id'] == $_SESSION['admin_user_id']): ?>
                                                <span class="ml-2 px-2 py-1 bg-blue-100 text-blue-800 text-xs rounded-full">You</span>
                                                <?php endif; ?>
                                            </div>
                                            <div class="text-xs text-gray-500">@<?= htmlspecialchars($user['username']) ?></div>
                                        </div>
                                    </div>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-600">
                                    <i class="ri-mail-line mr-1"></i><?= htmlspecialchars($user['email']) ?>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <?php if ($user['is_active']): ?>
                                    <span class="px-3 py-1 inline-flex text-xs leading-5 font-semibold rounded-full bg-green-100 text-green-800">
                                        <i class="ri-checkbox-circle-line mr-1"></i>Active
                                    </span>
                                    <?php else: ?>
                                    <span class="px-3 py-1 inline-flex text-xs leading-5 font-semibold rounded-full bg-red-100 text-red-800">
                                        <i class="ri-close-circle-line mr-1"></i>Inactive
                                    </span>
                                    <?php endif; ?>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-600">
                                    <?php if ($user['last_login']): ?>
                                    <i class="ri-time-line mr-1"></i><?= date('M d, Y H:i', strtotime($user['last_login'])) ?>
                                    <?php else: ?>
                                    <span class="text-gray-400">Never</span>
                                    <?php endif; ?>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-600">
                                    <div class="flex items-center space-x-2">
                                        <a href="?action=edit&id=<?= $user['id'] ?>"
                                           class="text-blue-600 hover:text-blue-800 font-medium">
                                            <i class="ri-edit-line mr-1"></i>Edit
                                        </a>
                                        <a href="?action=reset&id=<?= $user['id'] ?>"
                                           class="text-orange-600 hover:text-orange-800 font-medium">
                                            <i class="ri-key-line mr-1"></i>Reset
                                        </a>
                                        <?php if ($user['id'] != $_SESSION['admin_user_id']): ?>
                                        <button onclick="confirmDelete(<?= $user['id'] ?>, '<?= htmlspecialchars($user['username']) ?>')"
                                                class="text-red-600 hover:text-red-800 font-medium">
                                            <i class="ri-delete-bin-line mr-1"></i>Delete
                                        </button>
                                        <?php endif; ?>
                                    </div>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>

                <!-- Stats Footer -->
                <div class="bg-gray-50 px-6 py-4 border-t border-gray-200">
                    <p class="text-sm text-gray-600">
                        <i class="ri-team-line mr-1"></i>
                        Total: <strong><?= count($users) ?></strong> users
                        <span class="mx-2">•</span>
                        Active: <strong><?= count(array_filter($users, fn($u) => $u['is_active'])) ?></strong>
                    </p>
                </div>
            </div>

        <?php elseif ($action === 'add'): ?>

            <!-- Add User Form -->
            <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-8 max-w-2xl">
                <h2 class="text-2xl font-bold text-gray-800 mb-6 flex items-center">
                    <i class="ri-user-add-line text-blue-600 mr-2"></i>
                    Add New Admin User
                </h2>

                <form method="POST" action="" class="space-y-5">
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <div>
                            <label class="block text-gray-700 font-medium mb-2 text-sm">Username *</label>
                            <input type="text" name="username" required
                                   class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                                   placeholder="Min 3 karakter">
                        </div>

                        <div>
                            <label class="block text-gray-700 font-medium mb-2 text-sm">Full Name</label>
                            <input type="text" name="full_name"
                                   class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                                   placeholder="Enter full name">
                        </div>
                    </div>

                    <div>
                        <label class="block text-gray-700 font-medium mb-2 text-sm">Email *</label>
                        <input type="email" name="email" required
                               class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                               placeholder="admin@example.com">
                    </div>

                    <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <div>
                            <label class="block text-gray-700 font-medium mb-2 text-sm">Password *</label>
                            <input type="password" name="password" required
                                   class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                                   placeholder="Min 6 karakter">
                        </div>

                        <div>
                            <label class="block text-gray-700 font-medium mb-2 text-sm">Confirm Password *</label>
                            <input type="password" name="confirm_password" required
                                   class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                                   placeholder="Ulangi password">
                        </div>
                    </div>

                    <div class="flex items-center space-x-4 pt-4">
                        <button type="submit" name="add_user"
                                class="px-8 py-3 bg-blue-600 text-white rounded-lg font-semibold hover:bg-blue-700 transition-all">
                            <i class="ri-save-line mr-2"></i>Create User
                        </button>
                        <a href="users.php" class="px-8 py-3 bg-gray-200 text-gray-700 rounded-lg font-semibold hover:bg-gray-300 transition">
                            <i class="ri-arrow-left-line mr-2"></i>Cancel
                        </a>
                    </div>
                </form>
            </div>

        <?php elseif ($action === 'edit' && $userEdit): ?>

            <!-- Edit User Form -->
            <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-8 max-w-2xl">
                <h2 class="text-2xl font-bold text-gray-800 mb-6 flex items-center">
                    <i class="ri-user-settings-line text-blue-600 mr-2"></i>
                    Edit User: <?= htmlspecialchars($userEdit['username']) ?>
                </h2>

                <form method="POST" action="" class="space-y-5">
                    <input type="hidden" name="user_id" value="<?= $userEdit['id'] ?>">

                    <div>
                        <label class="block text-gray-700 font-medium mb-2 text-sm">Username</label>
                        <input type="text" value="<?= htmlspecialchars($userEdit['username']) ?>" disabled
                               class="w-full px-4 py-3 border border-gray-300 rounded-lg bg-gray-100 cursor-not-allowed">
                        <p class="text-xs text-gray-500 mt-1">Username tidak bisa diubah</p>
                    </div>

                    <div>
                        <label class="block text-gray-700 font-medium mb-2 text-sm">Full Name</label>
                        <input type="text" name="full_name" value="<?= htmlspecialchars($userEdit['full_name'] ?? '') ?>"
                               class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                               placeholder="Enter full name">
                    </div>

                    <div>
                        <label class="block text-gray-700 font-medium mb-2 text-sm">Email *</label>
                        <input type="email" name="email" value="<?= htmlspecialchars($userEdit['email']) ?>" required
                               class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                               placeholder="admin@example.com">
                    </div>

                    <div>
                        <label class="flex items-center cursor-pointer">
                            <input type="checkbox" name="is_active" value="1" <?= $userEdit['is_active'] ? 'checked' : '' ?>
                                   <?= $userEdit['id'] == $_SESSION['admin_user_id'] ? 'disabled' : '' ?>
                                   class="w-4 h-4 text-blue-600 border-gray-300 rounded focus:ring-blue-500">
                            <span class="ml-2 text-gray-700">Active user</span>
                        </label>
                        <?php if ($userEdit['id'] == $_SESSION['admin_user_id']): ?>
                        <p class="text-xs text-gray-500 mt-1">Tidak bisa menonaktifkan akun sendiri</p>
                        <?php endif; ?>
                    </div>

                    <div class="flex items-center space-x-4 pt-4">
                        <button type="submit" name="edit_user"
                                class="px-8 py-3 bg-blue-600 text-white rounded-lg font-semibold hover:bg-blue-700 transition-all">
                            <i class="ri-save-line mr-2"></i>Save Changes
                        </button>
                        <a href="users.php" class="px-8 py-3 bg-gray-200 text-gray-700 rounded-lg font-semibold hover:bg-gray-300 transition">
                            <i class="ri-arrow-left-line mr-2"></i>Cancel
                        </a>
                    </div>
                </form>
            </div>

        <?php elseif ($action === 'reset' && $userId): ?>

            <?php
            $stmt = $pdo->prepare("SELECT * FROM admin_users WHERE id = ?");
            $stmt->execute([$userId]);
            $userReset = $stmt->fetch();
            ?>

            <!-- Reset Password Form -->
            <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-8 max-w-2xl">
                <h2 class="text-2xl font-bold text-gray-800 mb-6 flex items-center">
                    <i class="ri-key-line text-orange-600 mr-2"></i>
                    Reset Password: <?= htmlspecialchars($userReset['username']) ?>
                </h2>

                <div class="bg-orange-50 border-l-4 border-orange-500 p-4 mb-6">
                    <p class="text-orange-800 text-sm">
                        <i class="ri-alert-line mr-2"></i>
                        User akan menggunakan password baru ini untuk login selanjutnya.
                    </p>
                </div>

                <form method="POST" action="" class="space-y-5">
                    <input type="hidden" name="user_id" value="<?= $userId ?>">

                    <div>
                        <label class="block text-gray-700 font-medium mb-2 text-sm">New Password *</label>
                        <input type="password" name="new_password" required
                               class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                               placeholder="Min 6 karakter">
                    </div>

                    <div>
                        <label class="block text-gray-700 font-medium mb-2 text-sm">Confirm Password *</label>
                        <input type="password" name="confirm_password" required
                               class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                               placeholder="Ulangi password baru">
                    </div>

                    <div class="flex items-center space-x-4 pt-4">
                        <button type="submit" name="reset_password"
                                class="px-8 py-3 bg-orange-600 text-white rounded-lg font-semibold hover:bg-orange-700 transition-all">
                            <i class="ri-key-line mr-2"></i>Reset Password
                        </button>
                        <a href="users.php" class="px-8 py-3 bg-gray-200 text-gray-700 rounded-lg font-semibold hover:bg-gray-300 transition">
                            <i class="ri-arrow-left-line mr-2"></i>Cancel
                        </a>
                    </div>
                </form>
            </div>

        <?php endif; ?>

    </main>

    <!-- Delete Confirmation Form (Hidden) -->
    <form id="deleteForm" method="POST" action="" style="display: none;">
        <input type="hidden" name="user_id" id="deleteUserId">
        <input type="hidden" name="delete_user" value="1">
    </form>

    <script>
        function confirmDelete(userId, username) {
            if (confirm(`Yakin ingin menghapus user "${username}"?\n\nTindakan ini tidak bisa dibatalkan!`)) {
                document.getElementById('deleteUserId').value = userId;
                document.getElementById('deleteForm').submit();
            }
        }
    </script>

</body>
</html>
