<?php
/**
 * PROFILE PAGE
 * View and edit admin user profile
 */

define('ADMIN_ACCESS', true);
require_once 'config/database.php';
require_once 'config/functions.php';

startSecureSession();
requireLogin();

$pdo = getDBConnection();
$userId = $_SESSION['admin_user_id'];
$error = '';
$success = '';

// Handle profile update
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_profile'])) {
    $fullName = sanitize($_POST['full_name'] ?? '');
    $email = sanitize($_POST['email'] ?? '');

    if (empty($email)) {
        $error = 'Email harus diisi!';
    } elseif (!isValidEmail($email)) {
        $error = 'Format email tidak valid!';
    } else {
        try {
            $stmt = $pdo->prepare("UPDATE admin_users SET full_name = ?, email = ? WHERE id = ?");
            $stmt->execute([$fullName, $email, $userId]);
            setFlashMessage('success', 'Profile berhasil diupdate!');
            header('Location: profile.php');
            exit;
        } catch (PDOException $e) {
            if ($e->getCode() == 23000) {
                $error = 'Email sudah digunakan oleh user lain!';
            } else {
                $error = 'Terjadi kesalahan: ' . $e->getMessage();
            }
        }
    }
}

// Handle password change
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['change_password'])) {
    $currentPassword = $_POST['current_password'] ?? '';
    $newPassword = $_POST['new_password'] ?? '';
    $confirmPassword = $_POST['confirm_password'] ?? '';

    if (empty($currentPassword) || empty($newPassword) || empty($confirmPassword)) {
        $error = 'Semua field password harus diisi!';
    } elseif (strlen($newPassword) < 6) {
        $error = 'Password baru minimal 6 karakter!';
    } elseif ($newPassword !== $confirmPassword) {
        $error = 'Password baru dan konfirmasi tidak cocok!';
    } else {
        // Verify current password
        $stmt = $pdo->prepare("SELECT password FROM admin_users WHERE id = ?");
        $stmt->execute([$userId]);
        $user = $stmt->fetch();

        if (!verifyPassword($currentPassword, $user['password'])) {
            $error = 'Password saat ini salah!';
        } else {
            $hashedPassword = hashPassword($newPassword);
            $stmt = $pdo->prepare("UPDATE admin_users SET password = ? WHERE id = ?");
            $stmt->execute([$hashedPassword, $userId]);

            setFlashMessage('success', 'Password berhasil diubah!');
            header('Location: profile.php');
            exit;
        }
    }
}

// Get current user data
$stmt = $pdo->prepare("SELECT * FROM admin_users WHERE id = ?");
$stmt->execute([$userId]);
$user = $stmt->fetch();

$pageTitle = 'My Profile';
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= $pageTitle ?> - Admin Panel</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://cdn.jsdelivr.net/npm/remixicon@3.5.0/fonts/remixicon.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Saira:wght@400;500;600;700&family=Poppins:wght@300;400;500;600&display=swap" rel="stylesheet">
    <style>
        * { font-family: 'Poppins', sans-serif; }
        h1, h2, h3 { font-family: 'Saira', sans-serif; }
    </style>
</head>
<body class="bg-gray-50">

    <?php require_once 'includes/navbar.php'; ?>

    <main class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">

        <!-- Page Header -->
        <div class="mb-8">
            <h1 class="text-3xl font-bold text-gray-900 flex items-center">
                <i class="ri-user-settings-line text-blue-600 mr-3"></i>
                My Profile
            </h1>
            <p class="text-gray-600 mt-2">Manage your account settings and password</p>
        </div>

        <!-- Flash Messages -->
        <?php if ($message = getFlashMessage()): ?>
            <div class="mb-6 p-4 rounded-lg border-l-4 <?= $message['type'] === 'success' ? 'bg-green-50 border-green-500 text-green-800' : 'bg-red-50 border-red-500 text-red-800' ?>">
                <div class="flex items-center">
                    <i class="<?= $message['type'] === 'success' ? 'ri-checkbox-circle-line' : 'ri-error-warning-line' ?> text-2xl mr-3"></i>
                    <p class="font-medium"><?= htmlspecialchars($message['message']) ?></p>
                </div>
            </div>
        <?php endif; ?>

        <?php if ($error): ?>
            <div class="mb-6 p-4 rounded-lg border-l-4 bg-red-50 border-red-500 text-red-800">
                <div class="flex items-center">
                    <i class="ri-error-warning-line text-2xl mr-3"></i>
                    <p class="font-medium"><?= htmlspecialchars($error) ?></p>
                </div>
            </div>
        <?php endif; ?>

        <div class="grid grid-cols-1 lg:grid-cols-3 gap-6">

            <!-- Profile Info Card -->
            <div class="lg:col-span-1">
                <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
                    <div class="text-center">
                        <div class="w-24 h-24 rounded-full bg-blue-600 flex items-center justify-center mx-auto mb-4">
                            <span class="text-4xl font-bold text-white">
                                <?= strtoupper(substr($user['username'], 0, 1)) ?>
                            </span>
                        </div>
                        <h3 class="text-xl font-bold text-gray-900"><?= htmlspecialchars($user['full_name'] ?: $user['username']) ?></h3>
                        <p class="text-gray-600 text-sm">@<?= htmlspecialchars($user['username']) ?></p>
                        <div class="mt-4 pt-4 border-t border-gray-200">
                            <div class="text-sm text-gray-600 space-y-2">
                                <p><i class="ri-mail-line mr-2"></i><?= htmlspecialchars($user['email']) ?></p>
                                <p><i class="ri-calendar-line mr-2"></i>Joined <?= date('M d, Y', strtotime($user['created_at'])) ?></p>
                                <?php if ($user['last_login']): ?>
                                <p><i class="ri-time-line mr-2"></i>Last login: <?= date('M d, Y H:i', strtotime($user['last_login'])) ?></p>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Forms -->
            <div class="lg:col-span-2 space-y-6">

                <!-- Update Profile Form -->
                <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
                    <h2 class="text-xl font-bold text-gray-800 mb-6 flex items-center">
                        <i class="ri-profile-line text-blue-600 mr-2"></i>
                        Update Profile
                    </h2>

                    <form method="POST" action="" class="space-y-4">
                        <div>
                            <label class="block text-gray-700 font-medium mb-2 text-sm">Username</label>
                            <input type="text" value="<?= htmlspecialchars($user['username']) ?>" disabled
                                   class="w-full px-4 py-3 border border-gray-300 rounded-lg bg-gray-100 cursor-not-allowed">
                            <p class="text-xs text-gray-500 mt-1">Username tidak bisa diubah</p>
                        </div>

                        <div>
                            <label class="block text-gray-700 font-medium mb-2 text-sm">Full Name</label>
                            <input type="text" name="full_name" value="<?= htmlspecialchars($user['full_name'] ?? '') ?>"
                                   class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                                   placeholder="Enter your full name">
                        </div>

                        <div>
                            <label class="block text-gray-700 font-medium mb-2 text-sm">Email *</label>
                            <input type="email" name="email" value="<?= htmlspecialchars($user['email']) ?>" required
                                   class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                                   placeholder="admin@example.com">
                        </div>

                        <button type="submit" name="update_profile"
                                class="w-full bg-blue-600 text-white py-3 rounded-lg font-semibold hover:bg-blue-700 transition-all">
                            <i class="ri-save-line mr-2"></i>
                            Update Profile
                        </button>
                    </form>
                </div>

                <!-- Change Password Form -->
                <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
                    <h2 class="text-xl font-bold text-gray-800 mb-6 flex items-center">
                        <i class="ri-lock-password-line text-blue-600 mr-2"></i>
                        Change Password
                    </h2>

                    <form method="POST" action="" class="space-y-4">
                        <div>
                            <label class="block text-gray-700 font-medium mb-2 text-sm">Current Password *</label>
                            <input type="password" name="current_password" required
                                   class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                                   placeholder="Enter current password">
                        </div>

                        <div>
                            <label class="block text-gray-700 font-medium mb-2 text-sm">New Password *</label>
                            <input type="password" name="new_password" required
                                   class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                                   placeholder="Minimal 6 karakter">
                        </div>

                        <div>
                            <label class="block text-gray-700 font-medium mb-2 text-sm">Confirm New Password *</label>
                            <input type="password" name="confirm_password" required
                                   class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                                   placeholder="Ulangi password baru">
                        </div>

                        <button type="submit" name="change_password"
                                class="w-full bg-red-600 text-white py-3 rounded-lg font-semibold hover:bg-red-700 transition-all">
                            <i class="ri-key-line mr-2"></i>
                            Change Password
                        </button>
                    </form>
                </div>

            </div>

        </div>

    </main>

</body>
</html>
