<?php
define('ADMIN_ACCESS', true);
require_once 'config/database.php';
require_once 'config/functions.php';

startSecureSession();
requireLogin();

$currentAdmin = getCurrentAdmin();
$pdo = getDBConnection();

$action = $_GET['action'] ?? 'list';
$gameId = $_GET['id'] ?? null;
$error = '';
$success = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['delete']) && $gameId) {
        $stmt = $pdo->prepare("DELETE FROM games WHERE id = ?");
        $stmt->execute([$gameId]);
        setFlashMessage('success', 'Game berhasil dihapus!');
        header('Location: games.php');
        exit;
    }

    if (isset($_POST['save'])) {
        $providerCode = sanitize($_POST['provider_code'] ?? '');
        $gameName = sanitize($_POST['game_name'] ?? '');
        $imagePath = sanitize($_POST['game_image_path'] ?? '');
        $displayOrder = (int)($_POST['display_order'] ?? 0);
        $isActive = isset($_POST['is_active']) ? 1 : 0;

        if (empty($providerCode) || empty($gameName)) {
            $error = 'Provider dan nama game harus diisi!';
        } else {
            if ($gameId) {
                $stmt = $pdo->prepare("UPDATE games SET provider_code = ?, game_name = ?, game_image_path = ?, display_order = ?, is_active = ?, updated_at = NOW() WHERE id = ?");
                $stmt->execute([$providerCode, $gameName, $imagePath, $displayOrder, $isActive, $gameId]);
                setFlashMessage('success', 'Game berhasil diupdate!');
            } else {
                $stmt = $pdo->prepare("INSERT INTO games (provider_code, game_name, game_image_path, display_order, is_active, created_at) VALUES (?, ?, ?, ?, ?, NOW())");
                $stmt->execute([$providerCode, $gameName, $imagePath, $displayOrder, $isActive]);
                setFlashMessage('success', 'Game berhasil ditambahkan!');
            }
            header('Location: games.php');
            exit;
        }
    }
}

$searchQuery = $_GET['search'] ?? '';
$filterProvider = $_GET['provider'] ?? '';
$page = max(1, (int)($_GET['page'] ?? 1));
$limit = 50;
$offset = ($page - 1) * $limit;

$whereConditions = [];
$params = [];

if ($searchQuery) {
    $whereConditions[] = "game_name LIKE ?";
    $params[] = "%$searchQuery%";
}

if ($filterProvider) {
    $whereConditions[] = "provider_code = ?";
    $params[] = $filterProvider;
}

$whereSQL = $whereConditions ? 'WHERE ' . implode(' AND ', $whereConditions) : '';

$countStmt = $pdo->prepare("SELECT COUNT(*) as total FROM games $whereSQL");
$countStmt->execute($params);
$totalGames = $countStmt->fetch()['total'];
$totalPages = ceil($totalGames / $limit);

$stmt = $pdo->prepare("SELECT * FROM games $whereSQL ORDER BY display_order ASC, id DESC LIMIT $limit OFFSET $offset");
$stmt->execute($params);
$games = $stmt->fetchAll();

$providers = $pdo->query("SELECT * FROM providers WHERE is_active = 1 ORDER BY provider_name")->fetchAll();

if ($action === 'edit' && $gameId) {
    $stmt = $pdo->prepare("SELECT * FROM games WHERE id = ? LIMIT 1");
    $stmt->execute([$gameId]);
    $gameEdit = $stmt->fetch();
    if (!$gameEdit) {
        header('Location: games.php');
        exit;
    }
}

$flashMessage = getFlashMessage();
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Games Management - Admin Panel</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://cdn.jsdelivr.net/npm/remixicon@3.5.0/fonts/remixicon.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Saira:wght@400;500;600;700&family=Poppins:wght@300;400;500;600&display=swap" rel="stylesheet">
    <style>
        * { font-family: 'Poppins', sans-serif; }
        h1, h2, h3, h4 { font-family: 'Saira', sans-serif; }
    </style>
</head>
<body class="bg-gray-50">

    <?php require_once 'includes/navbar.php'; ?>

    <main class="ml-64 p-8">
        <div class="mb-8">
            <h1 class="text-3xl font-bold text-gray-800 mb-2">Games Management</h1>
            <p class="text-gray-600">Kelola semua game di database</p>
        </div>

        <?php if ($flashMessage): ?>
            <div class="mb-6">
                <?php if ($flashMessage['type'] === 'success'): ?>
                    <div class="bg-green-50 border-l-4 border-green-500 p-4 rounded-lg">
                        <div class="flex items-center">
                            <i class="ri-checkbox-circle-line text-2xl text-green-500 mr-3"></i>
                            <p class="text-green-800"><?= htmlspecialchars($flashMessage['message']) ?></p>
                        </div>
                    </div>
                <?php endif; ?>
            </div>
        <?php endif; ?>

        <?php if ($action === 'add' || $action === 'edit'): ?>
            <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-6 mb-6">
                <h2 class="text-xl font-bold text-gray-800 mb-6"><?= $action === 'edit' ? 'Edit Game' : 'Tambah Game Baru' ?></h2>

                <?php if ($error): ?>
                    <div class="bg-red-50 border-l-4 border-red-500 p-4 mb-6 rounded">
                        <p class="text-red-700"><?= htmlspecialchars($error) ?></p>
                    </div>
                <?php endif; ?>

                <form method="POST" class="space-y-4">
                    <div>
                        <label class="block text-gray-700 font-medium mb-2 text-sm">Provider *</label>
                        <select name="provider_code" required class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500">
                            <option value="">Pilih Provider</option>
                            <?php foreach ($providers as $prov): ?>
                                <option value="<?= $prov['provider_code'] ?>" <?= (isset($gameEdit) && $gameEdit['provider_code'] === $prov['provider_code']) ? 'selected' : '' ?>>
                                    <?= htmlspecialchars($prov['provider_name']) ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <div>
                        <label class="block text-gray-700 font-medium mb-2 text-sm">Nama Game *</label>
                        <input type="text" name="game_name" value="<?= isset($gameEdit) ? htmlspecialchars($gameEdit['game_name']) : '' ?>" required
                               class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                               placeholder="Contoh: Gates of Olympus">
                    </div>

                    <div>
                        <label class="block text-gray-700 font-medium mb-2 text-sm">Image Path *</label>
                        <input type="text" name="game_image_path" value="<?= isset($gameEdit) ? htmlspecialchars($gameEdit['game_image_path']) : '' ?>" required
                               class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                               placeholder="Contoh: img/games-pp/gates-of-olympus.webp">
                    </div>

                    <div>
                        <label class="block text-gray-700 font-medium mb-2 text-sm">Display Order (Posisi)</label>
                        <input type="number" name="display_order" value="<?= isset($gameEdit) ? $gameEdit['display_order'] : 999 ?>" min="0"
                               class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                               placeholder="999 (semakin kecil semakin atas)">
                        <p class="text-xs text-gray-500 mt-1">Semakin besar angka, semakin depan posisinya. 0 = default order.</p>
                    </div>

                    <div>
                        <label class="flex items-center cursor-pointer">
                            <input type="checkbox" name="is_active" value="1" <?= (!isset($gameEdit) || $gameEdit['is_active']) ? 'checked' : '' ?>
                                   class="w-4 h-4 text-blue-600 border-gray-300 rounded focus:ring-blue-500">
                            <span class="ml-2 text-gray-700">Game Aktif</span>
                        </label>
                    </div>

                    <div class="flex space-x-3 pt-4">
                        <button type="submit" name="save" class="px-6 py-3 bg-blue-600 text-white rounded-lg font-semibold hover:bg-blue-700 transition">
                            <i class="ri-save-line mr-2"></i> <?= $action === 'edit' ? 'Update Game' : 'Tambah Game' ?>
                        </button>
                        <a href="games.php" class="px-6 py-3 bg-gray-200 text-gray-700 rounded-lg font-semibold hover:bg-gray-300 transition">
                            <i class="ri-close-line mr-2"></i> Batal
                        </a>
                    </div>
                </form>
            </div>
        <?php else: ?>
            <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-6 mb-6">
                <div class="flex flex-col md:flex-row md:items-center md:justify-between gap-4 mb-6">
                    <div class="flex-1">
                        <form method="GET" class="flex gap-3">
                            <input type="text" name="search" value="<?= htmlspecialchars($searchQuery) ?>" placeholder="Cari game..."
                                   class="flex-1 px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500">
                            <select name="provider" class="px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500">
                                <option value="">All Providers</option>
                                <?php foreach ($providers as $prov): ?>
                                    <option value="<?= $prov['provider_code'] ?>" <?= $filterProvider === $prov['provider_code'] ? 'selected' : '' ?>>
                                        <?= htmlspecialchars($prov['provider_name']) ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                            <button type="submit" class="px-6 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition">
                                <i class="ri-search-line"></i> Cari
                            </button>
                        </form>
                    </div>
                    <div class="flex space-x-3">
                        <a href="bulk-add-games.php" class="px-6 py-2 bg-orange-600 text-white rounded-lg hover:bg-orange-700 transition font-medium">
                            <i class="ri-file-list-3-line mr-2"></i> Bulk Add
                        </a>
                        <a href="games.php?action=add" class="px-6 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700 transition font-medium">
                            <i class="ri-add-line mr-2"></i> Tambah Game
                        </a>
                    </div>
                </div>

                <div class="overflow-x-auto">
                    <table class="w-full">
                        <thead class="bg-gray-50 border-b border-gray-200">
                            <tr>
                                <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">ID</th>
                                <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Image</th>
                                <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Provider</th>
                                <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Nama Game</th>
                                <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Order</th>
                                <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Status</th>
                                <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Aksi</th>
                            </tr>
                        </thead>
                        <tbody class="divide-y divide-gray-200">
                            <?php foreach ($games as $game): ?>
                                <tr class="hover:bg-gray-50">
                                    <td class="px-4 py-3 text-sm text-gray-900"><?= $game['id'] ?></td>
                                    <td class="px-4 py-3">
                                        <img src="../<?= htmlspecialchars($game['game_image_path']) ?>"
                                             alt="<?= htmlspecialchars($game['game_name']) ?>"
                                             class="w-12 h-12 object-cover rounded-lg shadow-sm border border-gray-200"
                                             onerror="this.src='data:image/svg+xml,<svg xmlns=%22http://www.w3.org/2000/svg%22 width=%2248%22 height=%2248%22 viewBox=%220 0 24 24%22 fill=%22%23ccc%22><rect width=%2224%22 height=%2224%22 fill=%22%23f3f4f6%22/><path d=%22M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z%22 stroke=%22%23999%22 stroke-width=%221%22 fill=%22none%22/></svg>'">
                                    </td>
                                    <td class="px-4 py-3 text-sm font-medium text-gray-900">
                                        <span class="px-2 py-1 bg-blue-100 text-blue-800 rounded text-xs font-medium"><?= htmlspecialchars($game['provider_code']) ?></span>
                                    </td>
                                    <td class="px-4 py-3 text-sm text-gray-900 max-w-xs">
                                        <div class="font-medium"><?= htmlspecialchars($game['game_name']) ?></div>
                                        <div class="text-xs text-gray-500 truncate" title="<?= htmlspecialchars($game['game_image_path']) ?>"><?= htmlspecialchars($game['game_image_path']) ?></div>
                                    </td>
                                    <td class="px-4 py-3 text-sm text-gray-900">
                                        <span class="px-2 py-1 bg-gray-100 text-gray-800 rounded text-xs font-medium"><?= $game['display_order'] ?></span>
                                    </td>
                                    <td class="px-4 py-3 text-sm">
                                        <?php if ($game['is_active']): ?>
                                            <span class="px-2 py-1 bg-green-100 text-green-800 rounded text-xs font-medium">Aktif</span>
                                        <?php else: ?>
                                            <span class="px-2 py-1 bg-red-100 text-red-800 rounded text-xs font-medium">Nonaktif</span>
                                        <?php endif; ?>
                                    </td>
                                    <td class="px-4 py-3 text-sm space-x-2">
                                        <a href="games.php?action=edit&id=<?= $game['id'] ?>" class="text-blue-600 hover:text-blue-800">
                                            <i class="ri-edit-line"></i> Edit
                                        </a>
                                        <form method="POST" action="games.php?id=<?= $game['id'] ?>" class="inline" onsubmit="return confirm('Yakin hapus game ini?')">
                                            <button type="submit" name="delete" class="text-red-600 hover:text-red-800">
                                                <i class="ri-delete-bin-line"></i> Hapus
                                            </button>
                                        </form>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>

                <?php if ($totalPages > 1): ?>
                    <div class="flex justify-center mt-6 space-x-2">
                        <?php for ($i = 1; $i <= min($totalPages, 10); $i++): ?>
                            <a href="?page=<?= $i ?><?= $searchQuery ? '&search=' . urlencode($searchQuery) : '' ?><?= $filterProvider ? '&provider=' . urlencode($filterProvider) : '' ?>"
                               class="px-4 py-2 <?= $i === $page ? 'bg-blue-600 text-white' : 'bg-gray-200 text-gray-700 hover:bg-gray-300' ?> rounded-lg transition">
                                <?= $i ?>
                            </a>
                        <?php endfor; ?>
                    </div>
                <?php endif; ?>

                <p class="text-center text-gray-600 mt-4 text-sm">Total: <?= $totalGames ?> games</p>
            </div>
        <?php endif; ?>
    </main>

</body>
</html>
