<?php
define('ADMIN_ACCESS', true);
require_once 'config/database.php';
require_once 'config/functions.php';

startSecureSession();
requireLogin();

$currentAdmin = getCurrentAdmin();
$pdo = getDBConnection();

$success = false;
$errors = [];
$imported = 0;

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['bulk_add'])) {
    $providerCode = strtoupper(sanitize($_POST['provider_code'] ?? ''));
    $gamesData = $_POST['games_data'] ?? '';

    if (empty($providerCode) || empty($gamesData)) {
        $errors[] = 'Provider code dan data games harus diisi!';
    } else {
        // Parse games data (format: game_name|image_url per line)
        $lines = explode("\n", trim($gamesData));

        $pdo->beginTransaction();

        try {
            $stmt = $pdo->prepare("INSERT INTO games (provider_code, game_name, game_image_path, display_order, is_active, created_at) VALUES (?, ?, ?, 0, 1, NOW())");

            foreach ($lines as $line) {
                $line = trim($line);
                if (empty($line)) continue;

                $parts = explode('|', $line);
                if (count($parts) !== 2) {
                    $errors[] = "Format salah: $line";
                    continue;
                }

                $gameName = trim($parts[0]);
                $imageUrl = trim($parts[1]);

                if (empty($gameName) || empty($imageUrl)) {
                    continue;
                }

                $stmt->execute([$providerCode, $gameName, $imageUrl]);
                $imported++;
            }

            // Update total games count
            $countStmt = $pdo->prepare("SELECT COUNT(*) as total FROM games WHERE provider_code = ? AND is_active = 1");
            $countStmt->execute([$providerCode]);
            $totalGames = $countStmt->fetch()['total'];

            $updateStmt = $pdo->prepare("UPDATE providers SET total_games = ? WHERE provider_code = ?");
            $updateStmt->execute([$totalGames, $providerCode]);

            $pdo->commit();
            $success = true;
            setFlashMessage('success', "$imported games berhasil ditambahkan!");
        } catch (Exception $e) {
            $pdo->rollBack();
            $errors[] = "Error: " . $e->getMessage();
        }
    }
}

$providers = $pdo->query("SELECT * FROM providers WHERE is_active = 1 ORDER BY provider_name")->fetchAll();
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Bulk Add Games - Admin Panel</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://cdn.jsdelivr.net/npm/remixicon@3.5.0/fonts/remixicon.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Saira:wght@400;500;600;700&family=Poppins:wght@300;400;500;600&display=swap" rel="stylesheet">
    <style>
        * { font-family: 'Poppins', sans-serif; }
        h1, h2, h3, h4 { font-family: 'Saira', sans-serif; }
    </style>
</head>
<body class="bg-gray-50">

    <?php require_once 'includes/navbar.php'; ?>

    <main class="ml-64 p-8">
        <div class="mb-8">
            <h1 class="text-3xl font-bold text-gray-800 mb-2">Bulk Add Games</h1>
            <p class="text-gray-600">Tambah banyak game sekaligus dengan CDN image URL</p>
        </div>

        <?php if ($success): ?>
            <div class="bg-green-50 border-l-4 border-green-500 p-4 rounded-lg mb-6">
                <div class="flex items-center">
                    <i class="ri-checkbox-circle-line text-2xl text-green-500 mr-3"></i>
                    <div>
                        <p class="text-green-800 font-medium">Berhasil import <?= $imported ?> games!</p>
                    </div>
                </div>
            </div>
        <?php endif; ?>

        <?php if (!empty($errors)): ?>
            <div class="bg-red-50 border-l-4 border-red-500 p-4 rounded-lg mb-6">
                <p class="text-red-800 font-medium mb-2">Errors:</p>
                <ul class="text-sm text-red-700 space-y-1">
                    <?php foreach ($errors as $error): ?>
                        <li>• <?= htmlspecialchars($error) ?></li>
                    <?php endforeach; ?>
                </ul>
            </div>
        <?php endif; ?>

        <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
            <form method="POST" class="space-y-6">
                <div>
                    <label class="block text-gray-700 font-medium mb-2">Select Provider *</label>
                    <select name="provider_code" required class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500">
                        <option value="">Pilih Provider</option>
                        <?php foreach ($providers as $prov): ?>
                            <option value="<?= $prov['provider_code'] ?>"><?= htmlspecialchars($prov['provider_name']) ?> (<?= $prov['provider_code'] ?>)</option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <div>
                    <label class="block text-gray-700 font-medium mb-2">Games Data *</label>
                    <textarea name="games_data" rows="15" required
                              class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 font-mono text-sm"
                              placeholder="Format: game_name|image_url&#10;&#10;Contoh:&#10;Gates of Olympus|https://cdn.example.com/gates.webp&#10;Sweet Bonanza|https://cdn.example.com/bonanza.webp&#10;Starlight Princess|https://cdn.example.com/starlight.webp"></textarea>
                    <p class="text-xs text-gray-500 mt-2">
                        <strong>Format:</strong> Nama Game | URL Gambar (satu game per baris)<br>
                        <strong>Note:</strong> Gambar dari CDN akan di-resize otomatis di frontend
                    </p>
                </div>

                <div class="bg-blue-50 border-l-4 border-blue-500 p-4 rounded">
                    <p class="text-sm text-blue-800 font-medium mb-2">Tips:</p>
                    <ul class="text-sm text-blue-700 space-y-1">
                        <li>• Satu game per baris</li>
                        <li>• Format: <code class="bg-blue-100 px-2 py-0.5 rounded">Nama Game|URL Gambar</code></li>
                        <li>• Gunakan URL gambar dari CDN (https://...)</li>
                        <li>• Image akan otomatis resize di frontend</li>
                        <li>• Baris kosong akan di-skip</li>
                    </ul>
                </div>

                <div class="flex space-x-3">
                    <button type="submit" name="bulk_add" class="px-6 py-3 bg-green-600 text-white rounded-lg font-semibold hover:bg-green-700 transition">
                        <i class="ri-add-box-line mr-2"></i> Bulk Add Games
                    </button>
                    <a href="games.php" class="px-6 py-3 bg-gray-200 text-gray-700 rounded-lg font-semibold hover:bg-gray-300 transition">
                        <i class="ri-arrow-left-line mr-2"></i> Back to Games
                    </a>
                </div>
            </form>
        </div>
    </main>

</body>
</html>
